package org.danbrough.hb.commands

import com.github.ajalt.clikt.parameters.options.default
import com.github.ajalt.clikt.parameters.options.flag
import com.github.ajalt.clikt.parameters.options.option
import com.github.ajalt.clikt.parameters.options.required
import com.github.ajalt.clikt.parameters.types.boolean
import kotlinx.serialization.Serializable
import org.danbrough.hb.HBContext
import org.danbrough.hb.Peer
import org.danbrough.hb.log
import org.danbrough.krch.KrchContext


@Serializable
data class PeerFlags(val enabled: Boolean, val push: Boolean, val pull: Boolean) {
  companion object {
    const val FLAG_ENABLED = 1
    const val FLAG_PUSH = 2
    const val FLAG_PULL = 4
  }

  fun flags(): Int =
    (if (push) FLAG_PUSH else 0) or (if (pull) FLAG_PULL else 0) or (if (enabled) FLAG_ENABLED else 0)

}


fun HBContext.peerListCommand(name: String) = object : KrchContext.Command(name) {

  val extra: Boolean by option().boolean().default(false)

  override fun run() {
    hbDatabase.peers().forEach {
      log.info { "peer: ${it.asString()} extra:$extra" }
    }
  }
}


fun HBContext.peerDeleteCommand(name: String) = object : KrchContext.Command(name) {
  val name by option().required()
  override fun run() {
    hbDatabase.peerDelete(name)
  }
}

fun HBContext.peerAddCommand(name: String) = object : KrchContext.Command(name) {
  val peerName by option("--name").required()
  val url by option().required()
  val pull by option().flag()
  val push by option().flag()

  val enabled by option().flag()

  val password by option()

  override fun run(): Unit = runInForeground {
    log.debug { "peerName: $peerName url: $url pull:$pull push:$push enabled: $enabled password:${password != null}" }
    hbDatabase.queries.peerGetByName(peerName).executeAsOneOrNull()
      ?.also { log.trace { "peer: $peerName already exists" } } ?: run {
      val newFlags = PeerFlags(enabled, push, pull)
      log.trace { "${commandName}: pull: $pull push: $push enabled: $enabled newFlags:$newFlags" }
      val peer = Peer(0, peerName, url, 0, 0, newFlags, password)
      log.debug { "${commandName}: adding peer: ${peer.asString()}" }
      hbDatabase.peerAdd(peer).await().also {
        log.trace { "${commandName}: received: $it" }
      }
    }
  }
}


fun HBContext.peerUpdateCommand(name: String) = object : KrchContext.Command(name) {
  val name by option().required()

  val newName: String? by option()
  val url: String? by option()
  val pull by option().flag()
  val push by option().flag()
  val enabled by option().flag()
  val password: String? by option(hidden = true)

  override fun run() {
    val peer = hbDatabase.peerGetByName(name).executeAsOneOrNull() ?: error("No such peer: $name")

    val newFlags = PeerFlags(enabled, push, pull)
    log.trace { "newFlags:$newFlags" }

    peer.copy(
      name = newName ?: peer.name,
      url = url ?: peer.url,
      password = password ?: peer.password,
      flags = newFlags
    ).also {
      hbDatabase.peerUpdate(it)
    }
  }
}


fun Peer.asString() =
  "Peer<$id,$name,$url,localID:$localID,remoteID:$remoteID,password:${if (!password.isNullOrBlank()) "****" else "null"},flags:$flags>"


class PeerSession(private val context: HBContext, val peer: Peer) {
  suspend fun peerSaveLocalID(id: Long) {
    if (id != peer.localID) context.hbDatabase.peerUpdate(peer.copy(localID = id)).await()
  }
}

suspend fun withPeer(
  context: HBContext, peerName: String, block: suspend PeerSession.() -> Unit
) {
  val db = context.hbDatabase
  val peer = db.peerGetByName(peerName).executeAsOneOrNull() ?: error("peer $peerName not found")
  PeerSession(context, peer).block()
}

val Peer.wsUrl: String
  get() = "${url.replace("http", "ws")}/ws"


