package org.danbrough.hb

import kotlin.random.Random
import kotlin.time.Clock
import kotlin.time.Instant
import kotlin.uuid.Uuid


object UuidUtils {
  /**
   * Generates a new UUIDv7.
   * The first 48 bits are derived from the current Unix epoch timestamp in milliseconds.
   * The next 4 bits are set to the version (7).
   * The next 2 bits are set to the IETF variant (10).
   * The remaining bits are filled with cryptographically strong random data.
   *
   * @return A new UUIDv7.
   */
  fun generateUUIDv7(
    timestamp: Long = Clock.System.now().toEpochMilliseconds(),
    random: Random = Random.Default
  ): Uuid {
    // High 64 bits of the UUID
    var mostSigBits = 0L

    // Set the timestamp in the first 48 bits (6 bytes)
    mostSigBits =
      mostSigBits or ((timestamp and 0xFFFFFFFFFFFFL) shl 16) // Shift left by 16 bits to make space for version and variant

    // Set the version (7) in the next 4 bits
    mostSigBits = mostSigBits or (0x7L shl 12) // Version 7

    // Set the variant (IETF RFC 4122 variant, which is 10xx in binary) in the next 2 bits
    mostSigBits = mostSigBits or (0x2L shl 10) // Variant 10 (binary)

    // Fill the remaining bits of the most significant long with random data
    mostSigBits = mostSigBits or (random.nextLong() and 0x3FFL) // 10 random bits

    // Low 64 bits of the UUID - entirely random
    val leastSigBits = random.nextLong()

    return Uuid.fromLongs(mostSigBits, leastSigBits)
  }
}

val Uuid.time: Instant
  get() = toLongs { msb, _ -> Instant.fromEpochMilliseconds(msb shr 16) }

val Log.uuid: Uuid
  get() = Uuid.fromLongs(index1, index2)

