package org.danbrough.hb.commands

import app.cash.sqldelight.coroutines.asFlow
import app.cash.sqldelight.coroutines.mapToList
import com.github.ajalt.clikt.parameters.options.flag
import com.github.ajalt.clikt.parameters.options.option
import com.github.ajalt.clikt.parameters.options.required
import io.ktor.client.plugins.websocket.receiveDeserialized
import io.ktor.client.plugins.websocket.sendSerialized
import io.ktor.server.websocket.DefaultWebSocketServerSession
import io.ktor.server.websocket.receiveDeserialized
import io.ktor.server.websocket.sendSerialized
import io.ktor.websocket.CloseReason
import io.ktor.websocket.close
import kotlinx.coroutines.DelicateCoroutinesApi
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.currentCoroutineContext
import kotlinx.coroutines.flow.asFlow
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.flatMapMerge
import kotlinx.coroutines.flow.take
import kotlinx.coroutines.isActive
import org.danbrough.hb.HBContext
import org.danbrough.hb.log
import org.danbrough.hb.server.Message
import org.danbrough.krch.KrchContext
import kotlin.time.Duration
import kotlin.time.Duration.Companion.seconds

suspend fun DefaultWebSocketServerSession.serverSyncHandler(
  initialMessage: Message.Sync
) {
  log.info { "serverSyncHandler: $initialMessage" }
  while (isActive) {
    val message = receiveDeserialized<Message>()
    log.debug { "message: $message" }
    if (message !is Message.LogEntries) error("Expecting LogEntries not a ${message::class}")
    sendSerialized<Message>(Message.OK)
  }
}

@OptIn(ExperimentalCoroutinesApi::class)
suspend fun clientSyncToPeer(
  context: HBContext,
  peerName: String,
  once: Boolean = false,
  maxListSize: Int = 10,
  pollForChangesDuration: Duration = Duration.ZERO
) {
  log.info { "clientSyncToPeer(): peer:$peerName once:$once" }
  withPeer(context, peerName) {
    clientSession(context, peer) {
      sendSerialized<Message>(Message.Sync)
      val db = context.hbDatabase
      var lastID = peer.localID

      if (once) {
        db.queries.logEntriesById(lastID).asFlow()
          .mapToList(currentCoroutineContext()).take(1).flatMapMerge { it.chunked(10).asFlow() }
          .collect { logs ->
            log.debug { "sending ${logs.count()} logs from $lastID" }
            if (logs.isNotEmpty()) {
              sendSerialized<Message>(Message.LogEntries(logs))
              val ok = receiveDeserialized<Message>()
              if (ok != Message.OK) error("expecting OK not $ok")
              lastID = logs.last().id
              db.peerUpdate(peer.copy(localID = lastID))
            }
          }

        close(CloseReason(CloseReason.Codes.NORMAL, "Finished sync"))
        return@clientSession
      }

      db.latestLogID(pollForChangesDuration)
        .filter { id -> id > lastID }
        .flatMapMerge {
          db.queries.logEntriesById(lastID).asFlow()
            .mapToList(currentCoroutineContext()).flatMapMerge { it.chunked(maxListSize).asFlow() }
        }.collect { logs ->
          log.debug { "sending ${logs.count()} logs from $lastID" }
          sendSerialized<Message>(Message.LogEntries(logs))
          val ok = receiveDeserialized<Message>()
          if (ok != Message.OK) error("expecting OK not $ok")
          lastID = logs.last().id
          db.peerUpdate(peer.copy(localID = lastID))
        }
    }
  }
}

fun HBContext.syncCommand(name: String) = object : KrchContext.Command(name) {

  val peerName by option("--peer").required()
  val once by option().flag(default = false)
//  private var runJob: Job? = null

  @OptIn(DelicateCoroutinesApi::class)
  override fun run() {
    runInForeground {
      clientSyncToPeer(this@syncCommand, peerName, once, pollForChangesDuration = 5.seconds)
    }
  }


  /*  override fun onStop() {
      runJob?.also {
        log.info { "${this::class}::onStop() cancelling runJob" }
        it.cancel("App closing")
      }
    }*/
}


