package org.danbrough.hb.shared

import app.cash.sqldelight.TransacterImpl
import app.cash.sqldelight.db.AfterVersion
import app.cash.sqldelight.db.QueryResult
import app.cash.sqldelight.db.SqlDriver
import app.cash.sqldelight.db.SqlSchema
import kotlin.Long
import kotlin.Unit
import kotlin.reflect.KClass
import org.danbrough.hb.HBQueries
import org.danbrough.hb.Habitrack
import org.danbrough.hb.Peer

internal val KClass<Habitrack>.schema: SqlSchema<QueryResult.Value<Unit>>
  get() = HabitrackImpl.Schema

internal fun KClass<Habitrack>.newInstance(driver: SqlDriver, PeerAdapter: Peer.Adapter): Habitrack = HabitrackImpl(driver, PeerAdapter)

private class HabitrackImpl(
  driver: SqlDriver,
  PeerAdapter: Peer.Adapter,
) : TransacterImpl(driver),
    Habitrack {
  override val hBQueries: HBQueries = HBQueries(driver, PeerAdapter)

  public object Schema : SqlSchema<QueryResult.Value<Unit>> {
    override val version: Long
      get() = 8

    override fun create(driver: SqlDriver): QueryResult.Value<Unit> {
      driver.execute(null, """
          |CREATE TABLE Log (
          |    id INTEGER PRIMARY KEY AUTOINCREMENT ,
          |    index1 INTEGER NOT NULL, index2 INTEGER NOT NULL,
          |    subject TEXT NOT NULL,
          |    count INTEGER NOT NULL DEFAULT(1),
          |    node TEXT,
          |    UNIQUE (index1,index2)
          |)
          """.trimMargin(), 0)
      driver.execute(null, "CREATE TABLE  DBMetadata(id INTEGER PRIMARY KEY,version INTEGER NOT NULL )", 0)
      driver.execute(null, """
          |CREATE TABLE Peer(
          |    id INTEGER PRIMARY KEY AUTOINCREMENT,
          |    name TEXT NOT NULL UNIQUE,
          |    url TEXT NOT NULL,
          |    remoteID INTEGER NOT NULL DEFAULT (0),
          |    localID INTEGER NOT NULL DEFAULT (0),
          |    flags INTEGER NOT NULL DEFAULT (0),
          |    password TEXT
          |)
          """.trimMargin(), 0)
      driver.execute(null, "CREATE INDEX idx_subject ON Log (subject)", 0)
      driver.execute(null, "CREATE INDEX idx_node ON Log (node)", 0)
      driver.execute(null, "CREATE INDEX idx_index1 ON Log(index1)", 0)
      return QueryResult.Unit
    }

    private fun migrateInternal(
      driver: SqlDriver,
      oldVersion: Long,
      newVersion: Long,
    ): QueryResult.Value<Unit> {
      if (oldVersion <= 4 && newVersion > 4) {
        driver.execute(null, "ALTER TABLE Log RENAME \"type\" TO \"subject\"", 0)
      }
      if (oldVersion <= 5 && newVersion > 5) {
        driver.execute(null, "UPDATE Log SET subject = substring(subject,4) WHERE subject LIKE 'hb/%'", 0)
        driver.execute(null, "UPDATE Log SET subject = LOWER(subject)", 0)
      }
      if (oldVersion <= 6 && newVersion > 6) {
        driver.execute(null, """
            |CREATE TABLE Log2 (
            |    id INTEGER PRIMARY KEY AUTOINCREMENT ,
            |    index1 INTEGER NOT NULL, index2 INTEGER NOT NULL,
            |    subject TEXT NOT NULL,
            |    count INTEGER NOT NULL DEFAULT(1),
            |    node TEXT
            |)
            """.trimMargin(), 0)
        driver.execute(null, "INSERT INTO Log2(index1,index2,subject,count,node) SELECT index1,index2,subject,count,node FROM Log", 0)
        driver.execute(null, "DROP TABLE Log", 0)
        driver.execute(null, "ALTER TABLE Log2 RENAME TO Log", 0)
      }
      if (oldVersion <= 7 && newVersion > 7) {
        driver.execute(null, """
            |CREATE TABLE Peer(
            |    id INTEGER PRIMARY KEY AUTOINCREMENT,
            |    name TEXT NOT NULL UNIQUE,
            |    url TEXT NOT NULL,
            |    remoteID INTEGER NOT NULL DEFAULT (0),
            |    localID INTEGER NOT NULL DEFAULT (0),
            |    flags INTEGER NOT NULL DEFAULT (0)
            |)
            """.trimMargin(), 0)
        driver.execute(null, "CREATE UNIQUE INDEX IF NOT EXISTS log_index_idx  ON Log(index1,index2)", 0)
      }
      return QueryResult.Unit
    }

    override fun migrate(
      driver: SqlDriver,
      oldVersion: Long,
      newVersion: Long,
      vararg callbacks: AfterVersion,
    ): QueryResult.Value<Unit> {
      var lastVersion = oldVersion

      callbacks.filter { it.afterVersion in oldVersion until newVersion }
      .sortedBy { it.afterVersion }
      .forEach { callback ->
        migrateInternal(driver, oldVersion = lastVersion, newVersion = callback.afterVersion + 1)
        callback.block(driver)
        lastVersion = callback.afterVersion + 1
      }

      if (lastVersion < newVersion) {
        migrateInternal(driver, lastVersion, newVersion)
      }
      return QueryResult.Unit
    }
  }
}
