package org.danbrough.hb

import app.cash.sqldelight.driver.native.NativeSqliteDriver
import io.ktor.client.HttpClient
import io.ktor.client.HttpClientConfig
import io.ktor.client.engine.curl.Curl
import io.ktor.server.cio.CIO
import io.ktor.server.cio.CIOApplicationEngine
import io.ktor.server.engine.ApplicationEngine
import kotlinx.cinterop.toKString
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.IO
import org.danbrough.hb.server.KtorApplicationEngineFactory
import platform.posix.F_OK
import platform.posix.access
import platform.posix.getenv

const val ENV_SSL_CAINFO = "SSL_CAINFO"


class HBContextNative(scope: CoroutineScope) : HBContext(scope) {
  override fun getEnv(name: String): String? = getenv(name)?.toKString()
  override fun threadName(): String = platform.posix.pthread_self().toString()
  override val ioDispatcher: CoroutineDispatcher = Dispatchers.IO


  private val defaultCertFile: String by lazy {

    listOf(
      "/etc/ssl/certs/ca-bundle.crt",
      "/etc/ssl/certs/ca-certificates.crt",
    ).first {
      access(it, F_OK) == 0
    }
  }

  override fun internalCreateHttpClient(block: HttpClientConfig<*>.() -> Unit): HttpClient =
    HttpClient(Curl) {
      val caInfoPath = getenv(ENV_SSL_CAINFO)?.toKString()
      log.info { "caInfoPath: ${caInfoPath ?: "$ENV_SSL_CAINFO not set"}" }

      if (Platform.osFamily == OsFamily.LINUX || caInfoPath != null) {
        engine {
          caInfo = caInfoPath ?: defaultCertFile
          log.debug { "${this@HBContextNative::class}:: caInfo = $caInfo" }
        }
      }
      block()
    }

  @Suppress("UNCHECKED_CAST")
  override val serverEngine: KtorApplicationEngineFactory =
    CIO as KtorApplicationEngineFactory

  override fun configureServer(configuration: ApplicationEngine.Configuration) {
    configuration as CIOApplicationEngine.Configuration
    configuration.reuseAddress = true
  }

  override val hbDbProvider: SqlDriverProvider = {
    val db = rootCommand.dbPath
    log.warn { "dbPath: $db parentDIR: ${db.parent.toString()}" }

    NativeSqliteDriver(Habitrack.Schema, db.name, 4, onConfiguration = {
      it.copy(extendedConfig = it.extendedConfig.copy(basePath = db.parent!!.toString()))
    })
  }
}


/*
class HbNativeContext(scope: CoroutineScope, args: List<String>) :
  HBContextOld(scope, args.toMutableList()) {

  private val defaultCertFile: String by lazy {

    listOf(
      "/etc/ssl/certs/ca-bundle.crt",
      "/etc/ssl/certs/ca-certificates.crt",
    ).first {
      access(it, F_OK) == 0
    }.also {
      log.warn { "default cert file is: $it" }
    }
//    ).forEach { file ->
//      val result = access(file, F_OK)
//      org.example.project.test.NativeTests.Companion.log.info { "file: $file result: $result" }
//    }
  }

  override fun getEnv(name: String): String? = getenv(name)?.toKString()
  override fun threadName(): String = platform.posix.pthread_self().toString()

  override fun internalCreateHttpClient(block: HttpClientConfig<*>.() -> Unit): HttpClient =
    HttpClient(Curl) {
      val caInfoPath = getenv(ENV_SSL_CAINFO)?.toKString()
      log.info { "caInfoPath: ${caInfoPath ?: "$ENV_SSL_CAINFO not set"}" }

      if (Platform.osFamily == OsFamily.LINUX || caInfoPath != null) {
        engine {
          caInfo = caInfoPath ?: defaultCertFile
        }
      }
      block()
    }


  override val spaceXDbProvider: SqlDriverProvider = {
    NativeSqliteDriver(AppDatabase.Schema, "spacex.db", 4, onConfiguration = {
      it.copy(extendedConfig = it.extendedConfig.copy(basePath = "/tmp/"))
    })
  }

  @Suppress("UNCHECKED_CAST")
  override val serverEngine: KtorApplicationEngineFactory =
    CIO as KtorApplicationEngineFactory

  override val hbDbProvider: SqlDriverProvider = {
    val db = dbPath
    log.warn { "dbPath: $db parentDIR: ${db.parent.toString()}" }

    NativeSqliteDriver(Habitrack.Schema, db.name, 4, onConfiguration = {
      it.copy(extendedConfig = it.extendedConfig.copy(basePath = db.parent!!.toString()))
    })
  }

  override val ioDispatcher: CoroutineDispatcher
    get() = Dispatchers.IO

  override fun configureServer(configuration: ApplicationEngine.Configuration) {
    configuration as CIOApplicationEngine.Configuration
    configuration.reuseAddress = true

  }

  override fun runInForeground(block: suspend CoroutineScope.() -> Unit) =
    runBlocking(block = block)

  init {
    loadConfig()
  }
}
*/
