package org.danbrough.hb

import com.github.ajalt.clikt.parameters.options.default
import com.github.ajalt.clikt.parameters.options.option
import io.ktor.client.HttpClient
import io.ktor.client.HttpClientConfig
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.plugins.websocket.WebSockets
import io.ktor.serialization.kotlinx.KotlinxWebsocketSerializationConverter
import io.ktor.serialization.kotlinx.json.json
import io.ktor.server.application.Application
import io.ktor.server.engine.ApplicationEngine
import io.ktor.server.engine.EmbeddedServer
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import kotlinx.io.files.Path
import kotlinx.io.files.SystemFileSystem
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.JsonBuilder
import kotlinx.serialization.modules.serializersModuleOf
import org.danbrough.hb.server.KtorApplicationEngineFactory
import org.danbrough.hb.server.LogSerializer
import org.danbrough.krch.KrchContext

abstract class HBContext(scope: CoroutineScope) : KrchContext<HBContext.HBRootCommand>(
  scope,
  envVarPrefix = "HB",
  rootCommandName = "hb",
  defaultConfigName = "hb.properties",
  defaultHomeDirName = ".habitrack"
) {
  abstract val ioDispatcher: CoroutineDispatcher
  abstract val serverEngine: KtorApplicationEngineFactory

  open val configureJson: JsonBuilder.() -> Unit = {
    classDiscriminator = "type"
    prettyPrint = false
    serializersModule = serializersModuleOf(Log::class, LogSerializer)
  }

  val json: Json by lazy {
    Json(builderAction = configureJson)
  }

  internal abstract fun internalCreateHttpClient(block: HttpClientConfig<*>.() -> Unit): HttpClient

  open val configureApplication: (Application.() -> Unit)? = null

  open val configureHttpClient: HttpClientConfig<*>.() -> Unit = {

    install(ContentNegotiation) {
      json(this@HBContext.json)
    }

    install(WebSockets) {
      contentConverter = KotlinxWebsocketSerializationConverter(json)
      maxFrameSize = 1024 * 1024
    }
  }

  fun createHttpClient(block: HttpClientConfig<*>.() -> Unit = {}): HttpClient =
    internalCreateHttpClient {
      configureHttpClient()
      block()
    }


  abstract fun configureServer(configuration: ApplicationEngine.Configuration)


  open inner class HBRootCommand : RootCommand() {
    val adminPassword: String? by option()
    val nodeID: String by option().default("test")

    private val internalDBPath: String? by option(
      "--db", help = $$"""
      Path to the database file. $$${envVarPrefix}_DB.
      Either relative to the config file path or absolute.
    """
    )

    //TODO: Fix this
    val dbPath: Path by lazy {
      val configPath = configFilePath
      var path = Path(internalDBPath ?: "${configPath.name.replace(".properties", "")}.db")
      if (!path.isAbsolute) path = Path(configPath.parent!!, path.toString())

      path.also {
        if (!SystemFileSystem.exists(path.parent!!)) SystemFileSystem.createDirectories(
          path.parent!!, false
        )
      }
    }
  }

  abstract val hbDbProvider: SqlDriverProvider

  val hbDatabase: HBDatabase by lazy { HBDatabase(hbDbProvider) }


  override val rootCommand: HBRootCommand = HBRootCommand()


  private var stopHandler: (() -> Unit) = {
    log.trace {
      "${this@HBContext::class.simpleName}: running stop handler: ${
        rootCommand.registeredSubcommands().joinToString(",") { it.commandName }
      }"
    }
    rootCommand.registeredSubcommands().forEach {
      if (it is Command)
        it.onStop()
    }
  }

  var running: Boolean = true
    set(value) {
      if (!value && field) {
        field = value
        stopHandler.invoke()
      } else field = value
    }

  fun onStop(handler: () -> Unit) {
    val oldHandler = stopHandler
    stopHandler = {
      handler()
      oldHandler.invoke()
    }
  }


}


//
//interface ServerConfig {
//  val port: Int
//  val bindAddress: String?
//}
//
//
//@Deprecated("Use HBContext instead")
//abstract class HBContextOld protected constructor(
//  val scope: CoroutineScope, val args: MutableList<String>
//) : CliktCommand() {
//  companion object {
//
//    const val ENV_VAR_PREFIX = "HB"
//
//  }
//
//  open val configureApplication: (Application.() -> Unit)? = null
//
//  internal val valueMap = mutableMapOf<String, String>()
//
//  internal val commands = mutableListOf<HabitrackCommand>()
//
//  init {
//    context {
//      autoEnvvarPrefix = ENV_VAR_PREFIX
//      terminal = Terminal(ansiLevel = AnsiLevel.TRUECOLOR, interactive = false)
//      valueSource = MapValueSource(valueMap)
//    }
//  }
//
//  override val allowMultipleSubcommands: Boolean = true
//
//  open val nodeID: String by option().default("test")
//
//
//  open val adminPassword: String? by option()
//
//  private val internalDBPath: String? by option(
//    "--db", help = $$"""
//      Path to the database file. $$${ENV_VAR_PREFIX}_DB.
//      Either relative to the config file path or absolute.
//    """
//  )
//
//  open val dbPath: Path by lazy {
//    val configPath = configFilePath
//    var path = Path(internalDBPath ?: "${configPath.name.replace(".properties", "")}.db")
//    if (!path.isAbsolute) path = Path(configPath.parent!!, path.toString())
//
//    path.also {
//      if (!SystemFileSystem.exists(path.parent!!)) SystemFileSystem.createDirectories(
//        path.parent!!, false
//      )
//    }
//  }
//
//
//  private val internalConfig: String? by option(
//    "--config", "-c", "-config", help = $$"Path to a config file. $$${ENV_VAR_PREFIX}_CONFIG"
//  )
//
//  open val configFilePath: Path by lazy {
//    (getEnv("${ENV_VAR_PREFIX}_CONFIG") ?: args.indexOfFirst { it.contains("--config") }
//      .takeIf { it > -1 }?.let { i ->
//        args[i].substringAfter('=').trim().also {
//          args.removeAt(i)
//        }
//      } ?: args.indexOf("-c").takeIf { it > -1 }?.let { i ->
//      args.removeAt(i)
//      args[i].also {
//        args.removeAt(i)
//      }
//    } ?: args.indexOf("-config").takeIf { it > -1 }?.let { i ->
//      args.removeAt(i)
//      args[i].also {
//        args.removeAt(i)
//      }
//    } ?: Path(getEnv("HOME") ?: ".", ".habitrack", "hb.properties").toString()).let {
//      val path = Path(it)
//      log.warn { "resolving $path" }
//      if (SystemFileSystem.exists(path)) SystemFileSystem.resolve(path) else path
//    }
//  }
//
//  override fun run() {
//    hbDatabase.runMigrations(Migration("FirstMigration", 0, 2) { migration, db ->
//      //  log.warn { "DOING migration: $migration" }
//    }, Migration("SecondMigration", 1, 3) { migration, db ->
//      //    log.warn { "DOING THIS TOO!" }
//    })
//
//    log.info { "dbPath:$dbPath nodeID:$nodeID adminPassword: $adminPassword" }
//  }
//
//
//  abstract fun getEnv(name: String): String?
//  abstract fun threadName(): String
//
//
//  open val configureJson: JsonBuilder.() -> Unit = {
//    classDiscriminator = "type"
//    prettyPrint = false
//    serializersModule = serializersModuleOf(Log::class, LogSerializer)
//  }
//
//  val json: Json by lazy {
//    Json(builderAction = configureJson)
//  }
//
//
//  open val configureHttpClient: HttpClientConfig<*>.() -> Unit = {
//
//    install(ContentNegotiation) {
//      json(this@HBContextOld.json)
//    }/*
//
//        install(DefaultRequest) {
//          header("Authorization", "Bearer $authSecret")
//        }
//    */
//
//    install(WebSockets) {
//      contentConverter = KotlinxWebsocketSerializationConverter(json)
//      maxFrameSize = 1024 * 1024
//    }
//  }
//
//  internal abstract fun internalCreateHttpClient(block: HttpClientConfig<*>.() -> Unit): HttpClient
//  fun createHttpClient(block: HttpClientConfig<*>.() -> Unit = {}): HttpClient =
//    internalCreateHttpClient {
//      configureHttpClient()
//      block()
//    }
//
//  abstract val hbDbProvider: SqlDriverProvider
//  abstract val spaceXDbProvider: SqlDriverProvider
//  abstract val serverEngine: KtorApplicationEngineFactory
//
//  abstract val ioDispatcher: CoroutineDispatcher
//
//  val spaceXSDK by lazy {
//    SpaceXSDK(this)
//  }
//
//  val hbDatabase: HBDatabase by lazy { HBDatabase(hbDbProvider) }
//
//  abstract fun configureServer(configuration: io.ktor.server.engine.ApplicationEngine.Configuration)
//
//  private var stopHandler: (() -> Unit) = {
//    commands.forEach(HabitrackCommand::onStop)
//  }
//
//  var running: Boolean = true
//    set(value) {
//      if (!value && field) {
//        field = value
//        stopHandler.invoke()
//      } else field = value
//    }
//
//  fun onStop(handler: () -> Unit) {
//    val oldHandler = stopHandler
//    stopHandler = {
//      handler()
//      oldHandler.invoke()
//    }
//  }
//
//  suspend fun waitForAllToFinish() {
//    log.debug { "waitForAllToFinish" }
//
//    commands.map { it.job }.filter { it != null }.forEach {
//      it!!.join()
//    }
//  }
//
//  abstract fun runInForeground(block: suspend CoroutineScope.() -> Unit)
//
//  open fun pathToSource(path: String): Source = SystemFileSystem.source(Path(path)).buffered()
//
//  open fun pathToSink(path: String): Sink = SystemFileSystem.sink(Path(path), false).buffered()
//
//
//  protected open fun loadConfig() {
//
//    error("shouldn't be called")
//    /*    log.info { "loadConfig(): configPath:'$configFilePath' args: ${args.joinToString(",")}" }
//
//        val extraArgs = mutableListOf<String>()
//
//        parseConfigFile(configFilePath).forEach {
//          log.trace { "config: ${if (it.first.contains("password")) "${it.first}=*****" else "${it.first}=${it.second}"}" }
//          if (it.second != null) valueMap[it.first] = it.second!!
//          else extraArgs += it.first
//        }
//
//        log.debug { "adding extraArgs: ${extraArgs.joinToString(",")} to args:${args.joinToString(",")}" }
//        args.addAll(0, extraArgs)*/
//  }
//
//
//}
//

