package org.danbrough.hb.server

import com.github.ajalt.clikt.parameters.options.default
import com.github.ajalt.clikt.parameters.options.flag
import com.github.ajalt.clikt.parameters.options.option
import com.github.ajalt.clikt.parameters.options.versionOption
import com.github.ajalt.clikt.parameters.types.boolean
import com.github.ajalt.clikt.parameters.types.int
import io.ktor.client.plugins.DefaultRequest
import io.ktor.client.request.header
import io.ktor.client.request.post
import io.ktor.client.statement.bodyAsText
import io.ktor.serialization.kotlinx.KotlinxWebsocketSerializationConverter
import io.ktor.serialization.kotlinx.json.json
import io.ktor.server.application.Application
import io.ktor.server.application.install
import io.ktor.server.auth.Authentication
import io.ktor.server.auth.authenticate
import io.ktor.server.engine.ApplicationEngine
import io.ktor.server.engine.ApplicationEngineFactory
import io.ktor.server.engine.EmbeddedServer
import io.ktor.server.engine.EngineConnectorBuilder
import io.ktor.server.engine.embeddedServer
import io.ktor.server.plugins.contentnegotiation.ContentNegotiation
import io.ktor.server.request.receive
import io.ktor.server.response.respond
import io.ktor.server.response.respondText
import io.ktor.server.routing.IgnoreTrailingSlash
import io.ktor.server.routing.get
import io.ktor.server.routing.post
import io.ktor.server.routing.route
import io.ktor.server.routing.routing
import io.ktor.server.websocket.WebSockets
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.Serializer
import org.danbrough.hb.HBContext
import org.danbrough.hb.JsonLog
import org.danbrough.hb.Log
import org.danbrough.hb.LogType
import org.danbrough.hb.jsonLog
import org.danbrough.hb.log
import org.danbrough.krch.KrchContext
import kotlin.time.Clock

const val DEFAULT_BIND_ADDRESS = "localhost"
const val DEFAULT_PORT = 8567

data class CustomPrincipal(val userName: String, val realm: String)
typealias KtorApplicationEngineFactory = ApplicationEngineFactory<ApplicationEngine, ApplicationEngine.Configuration>

@OptIn(ExperimentalSerializationApi::class)
@Serializer(forClass = Log::class)
object LogSerializer

fun HBContext.serverCommand(name: String): KrchContext.Command =
  object : KrchContext.Command(name) {
    private lateinit var server: EmbeddedServer<ApplicationEngine, ApplicationEngine.Configuration>
    val bindAddress: String by option().default(DEFAULT_BIND_ADDRESS)
    val port: Int by option().int().default(DEFAULT_PORT)
    val shutdown by option().flag()

    init {
      versionOption("0.0.1")
    }

    override fun run() {
      log.info { "shutdown: $shutdown" }
      if (shutdown) return runInForeground {
        log.info { "shutting down server at http://$bindAddress:$port/shutdown .." }
        createHttpClient {
          if (rootCommand.adminPassword != null) {
            install(DefaultRequest) {
              header("Authorization", "Bearer ${rootCommand.adminPassword}")
            }
          } else error("admin password not set")
        }.post("http://$bindAddress:$port/shutdown").bodyAsText().also {
          log.debug { it }
        }
      }


      runInBackground {
        log.info { "bindAddress:$bindAddress port:$port nodeID: ${rootCommand.nodeID}" }
        val serverPort = port
        server = embeddedServer(serverEngine, configure = {
          connectors.add(EngineConnectorBuilder().apply {
            this.host = bindAddress
            this.port = serverPort
          })
          connectionGroupSize = 2
          workerGroupSize = 5
          callGroupSize = 10
          shutdownGracePeriod = 1000
          shutdownTimeout = 1000

          configureServer(this)


        }, module = { hbModule(this@serverCommand) })
        /*server.addShutdownHook {
          log.info { "server shutdown hook" }
        }*/


        server.startSuspend(true)
      }
      log.warn { "${this::class}::run() finished" }
    }

    override fun onStop() {
      log.info { "$commandName::onStop()" }
      if (::server.isInitialized) {
        runInBackground {
          log.debug { "calling server.stop.." }
          server.stop()
          log.debug { "server.stop done" }
        }
      }
    }
  }


fun Application.hbModule(context: HBContext) {

  install(ContentNegotiation) {
    json(context.json)
  }

  install(IgnoreTrailingSlash)

  context.configureApplication?.invoke(this)

  context.rootCommand.adminPassword?.also { adminPassword ->
    install(Authentication) {
      configureAuthentication(adminPassword)
    }
  }

  install(WebSockets) {
    contentConverter = KotlinxWebsocketSerializationConverter(context.json)
    maxFrameSize = 1048576
  }

  routing {
    get("/test") {
      call.respond("Hello world at ${Clock.System.now()}!\n")
    }
    authenticate("admin", optional = false) {
      post("/shutdown") {
        call.respondText("Bye bye!\n")
        context.running = false
      }
    }

    authenticate("habitrack", optional = false) {

      webSocketHandler(context, "/ws")

      watchDataHandler(context, path = "/watch")

      post("/upload") {
        log.trace { "/upload post" }
        val entries = call.receive<List<JsonLog>>()
        log.trace { "/upload $entries" }
        context.hbDatabase.insertLogs(entries)
        call.respond("Thanks!")
      }

      post("/insert/{subject}") {
        val subject = call.parameters["subject"]!!
        val count = call.parameters["count"]?.toLong() ?: 1L

        log.trace { "insert: subject:<$subject> count:$count" }

        context.hbDatabase.queries.transaction {
          for (n in 1..count) context.hbDatabase.insertLog(
            type = LogType.valueOf(subject.uppercase()),
            node = context.rootCommand.nodeID,
            count = n,
          )
        }
        call.respond("inserted $subject")
      }
    }

    route("/node/{node}") {

      get {
        val node = call.parameters["node"]
        log.trace { "node: $node" }
        when (node) {
          "all" -> call.respond(context.hbDatabase.logEntries().executeAsList().map { it.jsonLog })

          else -> call.respond(
            context.hbDatabase.logEntriesByNode(node!!).executeAsList().map { it.jsonLog })
        }
      }

      get("/today/?$".toRegex()) {
        val node = call.parameters["node"]
        log.trace { "node: $node" }
        call.respond(context.hbDatabase.today().executeAsList().map { it.jsonLog })
      }
    }
  }
}

