package org.danbrough.hb

import app.cash.sqldelight.ColumnAdapter
import app.cash.sqldelight.TransactionWithoutReturn
import app.cash.sqldelight.coroutines.asFlow
import app.cash.sqldelight.coroutines.mapToOne
import kotlinx.coroutines.currentCoroutineContext
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.cancellable
import kotlinx.coroutines.flow.distinctUntilChanged
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.merge
import kotlinx.coroutines.isActive
import org.danbrough.hb.commands.PeerFlags
import kotlin.math.max
import kotlin.time.Duration
import kotlin.uuid.Uuid


class HBDatabase(driverProvider: SqlDriverProvider) {

  private val db =
    Habitrack(driverProvider(), Peer.Adapter(object : ColumnAdapter<PeerFlags, Long> {
      override fun decode(databaseValue: Long): PeerFlags =
        databaseValue.toInt().let { flags ->
          PeerFlags(
            (flags and PeerFlags.FLAG_ENABLED) != 0,
            (flags and PeerFlags.FLAG_PUSH) != 0,
            (flags and PeerFlags.FLAG_PULL) != 0
          )
        }

      override fun encode(value: PeerFlags): Long = value.flags().toLong()
    }))

  val queries = db.hBQueries

  fun logEntries() = queries.logEntries()

  fun logEntriesByNode(node: String) = queries.logEntriesByNode(node)

  fun jsonLogFlow(
    fromID: Long = 0L,
    emptyListEmissionDuration: Duration = Duration.ZERO
  ): Flow<List<Log>> {
    var startID: Long = fromID

    val dbFlow = flow {
      log.info { "jsonLogFlow() startID: $startID" }
      queries.logMaxId().asFlow().mapToOne(currentCoroutineContext())
        .filter { it.MAX != null && it.MAX > startID }
        .collect {
          val id = it.MAX!!
          log.info { "new max id: $id" }
          emit(queries.logEntriesById(startID).executeAsList())
          startID = id
        }
    }

    val flow = if (emptyListEmissionDuration == Duration.ZERO) dbFlow else merge(
      dbFlow,
      flow {
        while (currentCoroutineContext().isActive) {
          delay(emptyListEmissionDuration)
          emit(emptyList())
        }
      })
    return flow.cancellable()
  }


  fun latestLogID(pollDuration: Duration = Duration.ZERO): Flow<Long> {
    val maxIDFlow =
      queries.logMaxId().asFlow().map { it.executeAsOneOrNull()?.MAX ?: -1L }
    if (pollDuration == Duration.ZERO)
      return maxIDFlow

    val pollFlow = flow {
      var lastID: Long? = null
      while (currentCoroutineContext().isActive) {
        delay(pollDuration)
        queries.logMaxId().executeAsOneOrNull()?.MAX?.also {
          if (it != lastID) {
            emit(it)
            lastID = it
          }
        }
      }
    }

    return merge(maxIDFlow, pollFlow).distinctUntilChanged()
  }

  fun today() = fromTime(Utils.today)
  fun fromTime(startTimeMillis: Long) =
    queries.logEntriesByIndex1(0L or ((startTimeMillis and 0xFFFFFFFFFFFFL) shl 16))

  fun insertLog(type: LogType, node: String, count: Long? = null) =
    UuidUtils.generateUUIDv7(Utils.now()).toLongs { msb, lsb ->
      log.trace { "new log $msb - $lsb" }
      queries.insert(msb, lsb, type.path(), count ?: 1, node)
    }

  fun insertLog2(msb: Long, lsb: Long, type: LogType, node: String, count: Long? = null) =
    queries.insert(msb, lsb, type.path(), count ?: 1, node)

  fun insertLogs(logs: List<JsonLog>) {
    queries.transaction {
      logs.forEach { entry ->
        val uuid = Uuid.parse(entry.id)
        val (msg, lsb) = uuid.toLongs { msb, lsb -> Pair(msb, lsb) }
        insertLog2(msg, lsb, LogType.valueOf(entry.subject), entry.node!!, entry.count.toLong())
      }
    }
  }

  fun getDBMetadata(): DBMetadata = queries.transactionWithResult {
    queries.dbMetadata().executeAsOneOrNull() ?: run {
      DBMetadata(
        id = 0,
        version = 1,
      ).apply {
        queries.updateMetadata(this)
      }
    }
  }

  fun updateMetadataVersion(version: Int) =
    queries.updateMetadata(getDBMetadata().copy(version = version.toLong()))


  fun peers() = queries.peers().executeAsList()

  fun peerAdd(peer: Peer) =
    queries.peerAdd(peer.name, peer.url, peer.remoteID, peer.localID, peer.flags, peer.password)

  fun peerDelete(name: String) = queries.peerDelete(name)

  fun peerUpdate(peer: Peer) = queries.peerUpdate(peer)
  fun peerGetByName(name: String) = queries.peerGetByName(name)

  fun runMigrations(vararg migrationArgs: Migration) {
    val md = getDBMetadata()
    log.trace { "runMigrations() $md  migrations: ${migrationArgs.toList()}" }

    migrationArgs.toList()
      .filter { it.finalVersion > md.version }
      .sortedWith { a, b ->
        if (a.afterVersion == b.afterVersion) a.finalVersion - b.finalVersion else a.afterVersion - b.afterVersion
      }.takeIf { it.isNotEmpty() }?.let { migrations ->
        log.debug { "migrations: $migrations" }
        queries.transaction {
          var finalVersion = 0
          migrations.forEach { migration ->
            log.debug { "running migration: $migration" }
            migration.run(this@HBDatabase, this)
            finalVersion = max(finalVersion, migration.finalVersion)
          }
          updateMetadataVersion(finalVersion)
          log.info { "updated metadata version to $finalVersion" }
        }
      }
  }

  fun logLatest(): Log? = queries.logLatest().executeAsOneOrNull()


}

class Migration(
  val description: String,
  val afterVersion: Int,
  val finalVersion: Int,
  val migration: TransactionWithoutReturn.(Migration, HBDatabase) -> Unit
) {

  init {
    if (afterVersion >= finalVersion) error("afterVersion $afterVersion >= finalVersion $finalVersion")
  }

  fun run(db: HBDatabase, transaction: TransactionWithoutReturn) = transaction.migration(this, db)


  override fun toString(): String = "Migration[$afterVersion -> $finalVersion]: $description"
}