package org.danbrough.krch

import com.github.ajalt.clikt.core.CliktCommand
import com.github.ajalt.clikt.core.Context
import com.github.ajalt.clikt.core.context
import com.github.ajalt.clikt.core.terminal
import com.github.ajalt.clikt.parameters.options.flag
import com.github.ajalt.clikt.parameters.options.option
import com.github.ajalt.clikt.sources.MapValueSource
import com.github.ajalt.mordant.rendering.AnsiLevel
import com.github.ajalt.mordant.terminal.Terminal
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancel
import kotlinx.coroutines.joinAll
import kotlinx.coroutines.launch
import kotlinx.coroutines.runBlocking
import kotlinx.io.Sink
import kotlinx.io.Source
import kotlinx.io.buffered
import kotlinx.io.files.Path
import kotlinx.io.files.SystemFileSystem
import kotlin.coroutines.CoroutineContext
import kotlin.coroutines.EmptyCoroutineContext


abstract class KrchContext<R : KrchContext.Command>(
  val scope: CoroutineScope,
  val envVarPrefix: String = "KRCH",
  val rootCommandName: String = "krch",
  val defaultHomeDirName: String = ".krch",
  val defaultConfigName: String = "krch.properties",
  private val contextBuilder: Context.Builder.() -> Unit = {
    autoEnvvarPrefix = envVarPrefix
    terminal = Terminal(ansiLevel = AnsiLevel.TRUECOLOR, interactive = false)
  }
) {

  open val coroutineContextMain: CoroutineContext = Dispatchers.Unconfined
  open val coroutineContextIO: CoroutineContext = Dispatchers.Default
  open val coroutineContextDefault: CoroutineContext = EmptyCoroutineContext

  abstract fun getEnv(name: String): String?
  abstract fun threadName(): String

  internal val args: MutableList<String> = mutableListOf()

  internal val valueMap = mutableMapOf<String, String>()

  protected val closeables: MutableList<AutoCloseable> = mutableListOf()

  private var stopHandler: (() -> Unit) = {
    log.trace {
      "${this@KrchContext::class.simpleName}: running stop handler: ${
        rootCommand.registeredSubcommands().joinToString(",") { it.commandName }
      }"
    }
    closeables.forEach(AutoCloseable::close)
    closeables.clear()
    rootCommand.registeredSubcommands().forEach {
      if (it is Command)
        it.close()
    }
    jobs.forEach {
      it.cancel("${this::class.simpleName} is stopping")
    }
  }

  var running: Boolean = true
    set(value) {
      if (!value && field) {
        field = value
        stopHandler.invoke()
      } else field = value
    }

  fun registerStopHandler(handler: () -> Unit) {
    val oldHandler = stopHandler
    stopHandler = {
      handler()
      oldHandler.invoke()
    }
  }

  fun registerCloseable(closeable: AutoCloseable) {
    closeables.add(closeable)
  }

  open val configFilePath: Path by lazy {
    log.trace { "configFilePath: checking ${envVarPrefix}_CONFIG_PATH from environment.." }
    val path: Path = getEnv("${envVarPrefix}_CONFIG_PATH")?.let {
      log.debug { "FOUND CONFIG IN ENV: $it" }
      Path(it)
    } ?: args.indexOfFirst { it.contains("--config") }.takeIf { it > -1 }?.let { i ->
      Path(args[i].substringAfter('=').trim().also {
        args.removeAt(i)
      })
    } ?: args.indexOf("-c").takeIf { it > -1 }?.let { i ->
      args.removeAt(i)
      Path(args[i].also {
        args.removeAt(i)
      })
    } ?: args.indexOf("-config").takeIf { it > -1 }?.let { i ->
      args.removeAt(i)
      Path(args[i].also {
        args.removeAt(i)
      })
    } ?: args.takeIf { it.isNotEmpty() }?.first()?.let { Path(it) }
      ?.takeIf { SystemFileSystem.exists(it) && SystemFileSystem.metadataOrNull(it)?.isRegularFile == true }
      ?.also { args.removeAt(0) } ?: Path(
      getEnv("HOME") ?: ".", defaultHomeDirName, defaultConfigName
    )

    if (SystemFileSystem.exists(path)) SystemFileSystem.resolve(path) else path
  }

  private val jobs = mutableListOf<Job>()

  suspend fun waitForAllToFinish() {
    log.debug { "${this::class}::waitForAllToFinish()" }
    jobs.joinAll()
    log.debug { "${this::class}::waitForAllToFinish() finished" }
  }

  fun runInBackground(
    coroutineContext: CoroutineContext = coroutineContextDefault,
    block: suspend CoroutineScope.() -> Unit
  ) {
    jobs += scope.launch(coroutineContext, block = block)
  }

  fun <T> runInForeground(
    block: suspend CoroutineScope.() -> T
  ): T = runBlocking(block = block)


  abstract class Command(name: String) : CliktCommand(name), AutoCloseable {
    /**
     * Called when requested to stop running
     */
    override fun close() = Unit
  }

  /**
   * Top level command
   */
  open inner class RootCommand : Command(rootCommandName) {
    override val allowMultipleSubcommands = true
    val verbose by option().flag()

    init {
      context {
        contextBuilder()
        valueSource = MapValueSource(valueMap)
      }
    }

    override fun run() {
      log.debug { "$commandName::run() verbose:$verbose" }
    }
  }

  @Suppress("UNCHECKED_CAST")
  open val rootCommand: R = RootCommand() as R

  open fun pathToSource(path: String): Source = SystemFileSystem.source(Path(path)).buffered()

  open fun pathToSink(path: String): Sink = SystemFileSystem.sink(Path(path), false).buffered()

}
