package org.danbrough.krch.config

import kotlinx.io.buffered
import kotlinx.io.files.Path
import kotlinx.io.files.SystemFileSystem
import kotlinx.io.readLine

val log = klog.logger("KRCH_CONFIG")

enum class ConfigType {
  COMMAND, PROPERTY, FLAG
}

fun parseConfigFile(path: Path): Sequence<Triple<ConfigType, String, String?>> =
  sequence {

    log.debug { "parseConfigFile(): '$path'" }

    SystemFileSystem.source(SystemFileSystem.resolve(path)).buffered().use { source ->

      var name: String? = null
      var value: String? = null
      var lineNumber = 0

      while (true) {
        var line: String = source.readLine() ?: return@use
        lineNumber++

        if (value != null) {
          name ?: error("$path:$lineNumber: expected a name to be set")
          if (line.isBlank()) error("$path:$lineNumber: expecting more content for value $value")
          if (line.endsWith('\\')) {
            //continue with multi line
            value += line.dropLast(1)
            continue
          }
          //multi line value finished
          value += line.trim()

          yield(Triple(ConfigType.PROPERTY, name, value))
          name = null
          value = null
          continue
        }

        line = line.trim()
        log.trace { "$path:$lineNumber:line: '$line'" }
        if (line.startsWith('#')) {
          log.trace { "$path:$lineNumber: ignoring comment line: '$line'" }
          continue
        }

        if (line.startsWith("exec ")) {
          log.trace { "$path:$lineNumber: ignoring exec line: '$line'" }
          continue
        }

        if (line.isBlank()) {
          log.trace { "$path:$lineNumber: skipping blank line" }
          continue
        }

        if (line.startsWith('[')) {
          val i = line.indexOf(']')
          if (i == -1) error("$path:$lineNumber: expecting closing ']'")
          name = line.substring(1, i).trim().replace("\\n", "\n")
          val rest = line.substring(i + 1)
          if (rest.isNotBlank() && !rest.trim()
              .startsWith('#')
          ) error("$path:$lineNumber: extra content '$rest'")
          val alias = name.substringAfter(':')
          yield(Triple(ConfigType.COMMAND, name, alias.takeIf { it != name }))
          name = null
          continue
        }

        val n = line.indexOf('=')

        if (n > -1) {
          name = line.take(n).trim()
          log.trace { "$path:$lineNumber: found name: '$name'" }
          value = line.substring(n + 1).trim()
          val i = value.indexOf(" #")
          if (i > -1) value = value.take(i).trim()
          else if (value.endsWith('\\')) {
            value = value.dropLast(1)
            log.trace { "$path:$lineNumber: is multiline '$value'" }
            continue
          }
        } else {
          val flags = line.trim().split("\\s".toRegex())
          log.trace { "$path:$lineNumber: reading flags: ${flags.joinToString(",")}" }
          yield(Triple(ConfigType.FLAG, flags[0], null))
          //error("$path:Expecting name at line: $lineNumber")
          continue
        }

        yield(Triple(ConfigType.PROPERTY, name, value))
        name = null
        value = null
      }
    }

  }
