package org.danbrough.krch

import com.github.ajalt.clikt.core.main
import com.github.ajalt.clikt.core.subcommands
import kotlinx.io.files.SystemFileSystem
import org.danbrough.krch.config.ConfigType
import org.danbrough.krch.config.parseConfigFile

val log = klog.logger("KRCH")


fun KrchContext<*>.main(cmdLineArgs: Array<String>, commandMap: CommandMap) {
  log.info { "$this::main() cmdLineArgs: [${cmdLineArgs.joinToString(" ")}]" }


  //commandMap.commands().forEach { rootCommand.subcommands(it) }

  args.addAll(cmdLineArgs)

  loadConfig(commandMap)

  args.forEachIndexed { index, name ->
    if (name.startsWith('-')) return@forEachIndexed
    val cmdName = name.substringBefore(':')
    val alias = name.substringAfter(':')

    if (cmdName != alias) {
      log.warn { "commandMap.alias($cmdName, $alias)" }
      commandMap.alias(cmdName, alias)
      commandMap[alias]!!.also { command ->
        log.error { "adding command: $command" }
        rootCommand.subcommands(command)
        args[index] = alias
      }
    }
    /*commandMap[name]?.also { command ->
      rootCommand.subcommands(command)
      args[index] = command.commandName
      //log.trace { "replacing ${args[index]} with ${command.commandName}" }
    } ?: log.warn { "no command $name found in commandMap" }*/
  }
  log.trace { "modified args: $args" }
  log.trace {
    "registered commands: ${
      rootCommand.registeredSubcommands().joinToString(",") { it.commandName }
    }"
  }

  val commands = commandMap.commands()

  log.info { "commandMap commands: ${commands.joinToString(",") { it.commandName }}" }
  //log.info { "commandMap keys: ${commandMap.keys()}" }
  rootCommand.subcommands(commands)

  log.info { "calling: rootCommand.main(args) .." }
  rootCommand.main(args)
  log.info { "${this::class}.main(args) finished." }
}


typealias CommandConstructor = (String) -> KrchContext.Command

class CommandMap private constructor(private val map: MutableMap<String, CommandConstructor> = mutableMapOf()) {

  constructor(vararg commands: Pair<String, CommandConstructor>)
      : this(mutableMapOf(*commands))

  operator fun get(key: String): KrchContext.Command? =
    map[key.substringBefore(':')]?.invoke(key.substringAfter(':'))

  operator fun set(key: String, constructor: CommandConstructor) {
    map[key.substringAfter(':')] = constructor
  }

  fun commands(): List<KrchContext.Command> = map.keys.map { this[it]!! }

  fun alias(name: String, alias: String) {
    if (alias != name)
      map[alias] = map[name]!!
  }

  fun addAll(vararg commands: Pair<String, CommandConstructor>): CommandMap = apply {
    map.putAll(commands)
  }

}


private fun KrchContext<*>.loadConfig(commandMap: CommandMap) {

  if (!SystemFileSystem.exists(configFilePath)) {
    log.debug { "configPath $configFilePath not found" }
    return
  }

  log.debug { "loadConfig(): configPath:'$configFilePath'" }

  val extraArgs = mutableListOf<String>()

  var prefix = ""
  parseConfigFile(configFilePath).forEach { (type, key, value) ->
    log.debug { "config: $type: ${if (key.contains("password")) "$key=*****" else "$key=$value"}" }
    when (type) {
      ConfigType.PROPERTY -> {
        valueMap["$prefix$key"] = value!!
      }

      ConfigType.COMMAND -> {
        val alias = key.substringAfter(':')
        val cmdName = key.substringBefore(':')
        if (cmdName == "global") {
          prefix = ""
        } else {
          prefix = "$alias."
          log.trace { "adding alias: $cmdName -> $alias to extra args" }
          extraArgs += alias
          commandMap.alias(cmdName, alias)
          //rootCommand.subcommands(commandMap[alias]!!)
        }
      }

      ConfigType.FLAG -> {
        val flagValue = "$prefix${key.removePrefix("--")}"
        log.trace { "setting flag: $flagValue" }
        valueMap[flagValue] = "1"
      }
    }
    /*    if (value != null)
          valueMap["$prefix$key"] = value
        else {
          log.warn { "key:$key value:$value is null" }
          val alias = key.substringAfter(':')
          val cmdName = key.substringBefore(':')
          prefix = "$alias."
          log.trace { "adding alias: $cmdName -> $alias to extra args" }
          extraArgs += alias
          commandMap.alias(cmdName, alias)
          rootCommand.subcommands(commandMap[alias]!!)
        }*/
  }

  log.trace { "args: [${args.joinToString(" ")}] extra args: [${extraArgs.joinToString(" ")}]" }
  args.addAll(0, extraArgs)
}